/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

const float pi = 3.14159;

mat3 xrot(float t)
{
    return mat3(1.0, 0.0, 0.0,
                0.0, cos(t), -sin(t),
                0.0, sin(t), cos(t));
}

mat3 yrot(float t)
{
    return mat3(cos(t), 0.0, -sin(t),
                0.0, 1.0, 0.0,
                sin(t), 0.0, cos(t));
}

mat3 zrot(float t)
{
    return mat3(cos(t), -sin(t), 0.0,
                sin(t), cos(t), 0.0,
                0.0, 0.0, 1.0);
}

float pshade(vec3 p)
{
	 float ac = texture2D(iChannel0, vec2(p.y,p.z)).x;
    float bc = texture2D(iChannel0, vec2(p.x,p.z)).x;
    float cc = texture2D(iChannel0, vec2(p.x,p.y)).x;
    float s = ((ac + bc + cc) / 3.0) * 2.0 - 1.0;
    return s;
}

float sphere(vec3 p)
{
    vec3 q = fract(p+0.5) * 2.0 - 1.0;
	 return 1.3 - length(q);  
}

float map(vec3 p)
{
	return min(sphere(p), sphere(p+0.5));
}

float trace(vec3 o, vec3 r)
{
    float t = 0.0;
    for (int i = 0; i < 32; ++i) {
        vec3 p = o + r * t;
        float d = map(p);
        t += d * 0.5;
    }
    return t;
}

vec3 normal(vec3 p)
{
    vec3 o = vec3(0.01, 0.0, 0.0);
    return normalize(vec3(map(p+o.xyy) - map(p-o.xyy),
                          map(p+o.yxy) - map(p-o.yxy),
                          map(p+o.yyx) - map(p-o.yyx)));
}

vec3 pos(float time)
{
    vec3 o = vec3(0.25);

    float n = 3.0;
    float a = floor(time/n)*n;
    float b = fract(time/n)*n;
    
    o.x += a + clamp(b-0.0, 0.0, 1.0);
    o.y += a + clamp(b-1.0, 0.0, 1.0);
    o.z += a + clamp(b-2.0, 0.0, 1.0);
    
    return o;
}

vec3 smoothpos(float time)
{
	float a = floor(time);
    float b = fract(time);
    float c = smoothstep(0.0, 1.0, b);
    return pos(a+c);
}

float occlusion(vec3 origin, vec3 ray) {
    float delta = 0.1;
    const int samples = 16;
    float r = 0.0;
    for (int i = 1; i <= samples; ++i) {
        float t = delta * float(i);
     	vec3 pos = origin + ray * t;
        float dist = map(pos);
        float len = abs(t - dist);
        r += len * pow(2.0, -float(i));
    }
    return r;
}

vec4 surf(vec3 r, vec3 w, vec3 sn, float t)
{
    float prod = max(dot(sn,-r), 0.0);
    float off = 0.5 + 0.5 * sin(pshade(w)*pi*5.0);
    float fog = prod / (1.0 + t * t + off);
    return vec4(vec3(fog),off);
}

vec3 shade(vec3 o, vec3 r)
{
    float t = trace(o, r);
    vec3 w = o + r * t;
    vec3 sn = normal(w);
    
	float lit = occlusion(o, r) * 5.0;
    
    vec4 ac = surf(r, w, sn, t);
    
    vec3 from = vec3(0.8, 0.2, 0.1);
    vec3 to = vec3(1.0, 1.0, 1.0);
    
    float fx = 1.0 - ac.w;
    
    vec3 mixed = ac.xyz * mix(from, to, fx);
    
    vec3 fc = lit * mixed;
    
    return fc;
}

vec3 raydir(vec3 r, float t)
{
    return r * yrot(t) * xrot(t);
}

void main( void )
{
	vec2 uv = gl_FragCoord.xy / iResolution.xy;
    uv = uv * 2.0 - 1.0;
    uv.x *= iResolution.x / iResolution.y;
    
    vec3 r = normalize(vec3(uv, 1.0 - dot(uv,uv) * 0.5));
    vec3 o = vec3(0.0);
    
    float ms = iGlobalTime * 0.25;
    o.y = sin(ms*pi) * 0.25;
    o.z += ms;
    
    vec3 ao = smoothpos(ms);
    vec3 ar = raydir(r, ms);
    vec3 ac = shade(ao, ar);
    
    float of = 0.01;
    
    vec3 bo = smoothpos(ms-of);
    vec3 br = raydir(r, ms-of);
    vec3 bc = shade(bo, br);
    
    vec3 co = smoothpos(ms-of*2.0);
    vec3 cr = raydir(r, ms-of*2.0);
    vec3 cc = shade(co, cr);
    
    ac = ac * 0.5 + bc * 0.25 + cc * 0.25;
    
	gl_FragColor = vec4(ac, 1.0);
}